// Controller for Follow-Up Reminder Notifications (Updated for leads_comments table)
const mysqlConnection = require('../../utils/database');
const UnifiedNotificationService = require('../../services/NotificationService');
const FCMNotificationService = require('../../services/FCMNotificationService');

let localIO = null;

// Promisify database calls
function queryAsync(query, params = []) {
    return new Promise((resolve, reject) => {
        mysqlConnection.query(query, params, (err, results) => {
            if (err) reject(err);
            else resolve(results);
        });
    });
}

/**
 * Check for due follow-ups in leads_comments table and send notifications
 */
const checkDueFollowUps = async (ioInstance) => {
    const io = ioInstance || localIO;
    try {
        // Query for follow-ups that are due (using leads_comments table)
        // Check notified = 0 (unread) instead of nextfollowup = 1
        const query = `
      SELECT 
        lc.id as comment_id,
        lc.lead_id,
        lc.followupdate as scheduled_time,
        lc.comments as notes,
        lc.followup as followup_type,
        lc.user as user_name,
        u.id as user_id,
        u.email as user_email,
        CONCAT(u.first_name, ' ', u.last_name) AS full_name,
        cust.full_name as lead_name,
        cust.mobile as lead_phone
      FROM leads_comments lc
      INNER JOIN users u ON lc.user = u.name
      LEFT JOIN leads_main lm ON lc.lead_id = lm.id
      LEFT JOIN leads_customers cust ON lm.customer = cust.id
      WHERE lc.followupdate <= NOW()
        AND lc.followupdate IS NOT NULL
        AND lc.nextfollowup = 1
        AND (lc.notified IS NULL OR lc.notified = 0)
        AND lc.status = 'N'
      ORDER BY lc.followupdate ASC
    `;

        const dueFollowUps = await queryAsync(query);

        if (dueFollowUps.length === 0) {
            console.log('[Follow-Up Scheduler] No due follow-ups found');
            return { success: true, count: 0 };
        }

        console.log(`[Follow-Up Scheduler] Found ${dueFollowUps.length} due follow-ups`);

        // Process each follow-up
        for (const followUp of dueFollowUps) {
            try {
                const followUpData = {
                    id: followUp.lead_id,
                    name: followUp.lead_name,
                    notes: followUp.notes,
                    scheduled_time: followUp.scheduled_time,
                    lead_id: followUp.lead_id,
                };

                console.log(`📢 [Follow-Up] Sending notification to user ${followUp.user_id} (${followUp.user_name})`);

                // Send unified notification to all channels (ONLY to the assigned user)
                const result = await UnifiedNotificationService.sendFollowUpNotification(
                    followUp.user_id,
                    followUp.user_name,
                    followUpData
                );

                // Send FCM notification for follow-up (ONLY to the assigned user)
                const followupData = {
                    id: followUp.lead_id,
                    name: followUp.lead_name,
                    followUpNote: followUp.notes || followUp.scheduled_time,
                    type: 'followup_reminder'
                };

                await FCMNotificationService.sendFollowUpNotification(
                    followUp.user_id,
                    followupData
                ).catch(err => console.error('⚠️ FCM follow-up notification failed:', err.message));

                // Send via Socket.IO ONLY to the specific user (not admin or others)
                if (io) {
                    io.to(`user_${followUp.user_id}`).emit('notification', {
                        event: 'follow_up_reminder',
                        type: 'followup',
                        data: {
                            title: 'Follow-Up Reminder',
                            body: `Reminder: ${followupData.followUpNote || followupData.name}`,
                            message: `Reminder: ${followupData.followUpNote || followupData.name}`,
                            leadId: followupData.id,
                            lead_id: followupData.id,
                            userId: followUp.user_id,
                            user_id: followUp.user_id,
                            scheduledTime: followUp.scheduled_time,
                            created_at: new Date().toISOString(),
                            timestamp: new Date().toISOString()
                        }
                    });
                    console.log(`✅ [Follow-Up] Socket.IO notification sent ONLY to user ${followUp.user_id}`);
                }

                if (result.success) {
                    // Mark follow-up as notified (set notified = 1 for consistency)
                    const updateQuery = `
            UPDATE leads_comments 
            SET notified = 1
            WHERE id = ?
          `;
                    await queryAsync(updateQuery, [followUp.comment_id]);

                    // Insert into real_time_notifications table for history
                    try {
                        const insertNotificationQuery = `
              INSERT INTO real_time_notifications 
              (user_id, title, body, lead_id, notification_type, created_at) 
              VALUES (?, ?, ?, ?, ?, NOW())
            `;
                        await queryAsync(insertNotificationQuery, [
                            followUp.user_id,
                            'Follow-Up Reminder',
                            `Reminder: ${followupData.followUpNote || followupData.name}`,
                            followUp.lead_id,
                            'followup',
                        ]);
                        console.log(`[Follow-Up] ✓ Notification history saved for lead ${followUp.lead_id}`);
                    } catch (error) {
                        console.error('[Follow-Up] Error inserting notification history:', error.message);
                    }

                    console.log(`[Follow-Up] ✓ Reminder sent for comment ID: ${followUp.comment_id}`);
                } else {
                    console.error(`[Follow-Up] Failed to send notification:`, result.error);
                }
            } catch (error) {
                console.error(`[Follow-Up] Error processing comment ${followUp.comment_id}:`, error);
            }
        }

        return {
            success: true,
            count: dueFollowUps.length,
            message: `Processed ${dueFollowUps.length} follow-up reminders`,
        };
    } catch (error) {
        console.error('[Follow-Up Scheduler] Error checking due follow-ups:', error);
        return {
            success: false,
            error: error.message,
        };
    }
};

/**
 * Start the follow-up reminder scheduler
 * Checks every 1 minute for due follow-ups
 */
const startFollowUpScheduler = (io) => {
    console.log('[Follow-Up Scheduler] Starting...');
    localIO = io;

    // Run immediately on start
    checkDueFollowUps(io);

    // Then run every 1 minute (60000 ms)
    const interval = 1 * 60 * 1000; // 1 minute
    setInterval(() => {
        console.log('[Follow-Up Scheduler] Running scheduled check...');
        checkDueFollowUps(io);
    }, interval);

    console.log('[Follow-Up Scheduler] ✓ Started (checking every 1 minute)');
};

module.exports = {
    checkDueFollowUps,
    startFollowUpScheduler,
};
