const axios = require("axios");
const mysqlConnection = require('../../utils/database');
const { sendNotificationToUser } = require('../Dashboard/Notification');

/**
 * META WEBHOOK CONFIGURATION
 * 
 * 1. VERIFY_TOKEN: Any string you choose. It MUST match the "Verify Token" in Meta App Dashboard.
 * 2. META_PAGE_ACCESS_TOKEN: Required to fetch full lead data.
 *    How to get it:
 *    a. Go to Meta for Developers (https://developers.facebook.com/)
 *    b. Go to Tools -> Graph API Explorer
 *    c. Select your App in the top right.
 *    d. Under "User or Page", select the Facebook Page you want leads from.
 *    e. Add Permissions: 'leads_retrieval', 'pages_read_engagement', 'pages_manage_metadata'.
 *    f. Click "Generate Token".
 *    g. To make it never expire: Use the "Access Token Tool" to extend it to a "Long-lived token".
 */

const VERIFY_TOKEN = process.env.META_VERIFY_TOKEN || "Prosale_sale_app_elaanMarketing";
const META_GRAPH_VERSION = process.env.META_GRAPH_VERSION || "v18.0";
const META_PAGE_ACCESS_TOKEN = process.env.META_PAGE_ACCESS_TOKEN || "EAADpiuM8XJABQbDoVTkNlCZCCRfJC7c7w3SO0oey6NB0Bl6O6hdrsqv16vwlrZATSggHr1ZCFZBUdVSEV1VyTMx6wiWUdiJxUuXdswj4CzGv5zZARasi8UNlKZBhCrhfKqf5yjSsOz8ZCM1GxleVpCQwkkQJtCKWOSZCeT7TZAC2VsZAiWpeu6Ue85U3ZBtGpkDzHrFTLi18UijUxw2OLVvYnBHbDixeEyWRO2GZAPq5HnBKOQZAtWLrse227K8z1XSK6MieOmKF6hkZALveM4BeMQ9oah";

// Test mode: Set to false to actually process leads
const TEST_MODE = false;

/**
 * Webhook Verification Endpoint (GET)
 * Meta will call this to verify your webhook URL
 */
const verifyWebhook = (req, res) => {
    const mode = req.query["hub.mode"];
    const token = req.query["hub.verify_token"];
    const challenge = req.query["hub.challenge"];

    console.log("🔍 Webhook verification request received");
    console.log("Mode:", mode);
    console.log("Token:", token);

    if (mode === "subscribe" && token === VERIFY_TOKEN) {
        console.log("✅ Webhook verified successfully");
        return res.status(200).send(challenge);
    }

    console.log("❌ Webhook verification failed - token mismatch or invalid mode");
    console.log(`Expected Token: ${VERIFY_TOKEN}`);
    return res.status(403).send("Verification failed");
};

/**
 * Webhook Event Receiver (POST)
 * Receives lead data from Meta
 */
const receiveWebhook = async (req, res) => {
    try {
        console.log("\n****************************************");
        console.log("� POST REQUEST RECEIVED AT /webhook");
        console.log("****************************************");
        console.log("Time:", new Date().toISOString());
        console.log("Payload:", JSON.stringify(req.body, null, 2));

        // Always acknowledge quickly to Meta
        res.status(200).send("EVENT_RECEIVED");

        const body = req.body;
        if (body.object !== "page") return;

        for (const entry of body.entry || []) {
            for (const change of entry.changes || []) {
                if (change.field === "leadgen") {
                    const leadgenData = change.value;
                    console.log("🎯 Leadgen Event Received:", leadgenData.leadgen_id);
                    await processMetaLead(leadgenData);
                }
            }
        }
    } catch (error) {
        console.error("❌ Webhook error:", error);
    }
};

/**
 * Process Meta Lead Data
 */
const processMetaLead = async (leadgenData) => {
    try {
        console.log("\n========================================");
        console.log("🔄 Processing Meta Lead:", leadgenData.leadgen_id);

        if (!META_PAGE_ACCESS_TOKEN || META_PAGE_ACCESS_TOKEN.startsWith("EAAD...")) {
            console.log("❌ META_PAGE_ACCESS_TOKEN is invalid or missing.");
            console.log("Please update it in MetaWebhook.js or your .env file.");
            return;
        }

        // 1. Fetch full lead info using the token
        const leadDetails = await fetchLeadDetails(leadgenData.leadgen_id);

        // 2. Extract field_data into simple key:value object
        const extracted = extractFieldData(leadDetails.raw);

        console.log("📌 Extracted Lead Data:", JSON.stringify(extracted, null, 2));

        if (TEST_MODE) {
            console.log("⚠️ TEST MODE: Skip Database Save");
            return;
        }

        // 3. Save to database
        await saveLeadToDatabase(extracted, leadgenData);

    } catch (error) {
        console.error("❌ Error processing lead:", error?.response?.data || error.message);
    }
};

/**
 * Call Graph API to fetch lead details
 */
const fetchLeadDetails = async (leadgenId) => {
    const url = `https://graph.facebook.com/${META_GRAPH_VERSION}/${leadgenId}`;
    const params = { access_token: META_PAGE_ACCESS_TOKEN };
    const response = await axios.get(url, { params });
    return { raw: response.data };
};

/**
 * Convert Meta field_data array into a key:value object
 */
const extractFieldData = (leadApiResponse) => {
    const out = {};
    const fieldData = leadApiResponse.field_data || [];

    for (const item of fieldData) {
        const key = item.name;
        const value = Array.isArray(item.values) ? item.values[0] : item.values;
        out[key] = value ?? null;
    }

    out._lead_id = leadApiResponse.id;
    out._created_time = leadApiResponse.created_time;
    out._form_id = leadApiResponse.form_id;
    return out;
};

/**
 * Save Lead to Database
 */
const saveLeadToDatabase = async (leadData, metaRaw) => {
    const query = `
        INSERT INTO leads (name, email, phone, source, status, meta_lead_id, created_at) 
        VALUES (?, ?, ?, 'Meta Ads', 'New', ?, NOW())
    `;

    const values = [
        leadData.full_name || leadData.first_name || 'Meta Lead',
        leadData.email || null,
        leadData.phone_number || null,
        leadData._lead_id
    ];

    try {
        const [result] = await mysqlConnection.promise().execute(query, values);
        console.log("✅ Lead saved to database. ID:", result.insertId);

        // Optional: Send notification
        // sendNotificationToUser(1, "New Meta Lead Received", `New lead: ${leadData.email}`);
    } catch (err) {
        console.error("❌ DB Error:", err.message);
    }
};

module.exports = {
    verifyWebhook,
    receiveWebhook,
};

