const admin = require('../config/firebase');
const db = require('../utils/database');

// Wrapper to convert callback-based queries to promises
const queryAsync = (query, values) => {
  return new Promise((resolve, reject) => {
    db.query(query, values, (err, results) => {
      if (err) reject(err);
      else resolve(results);
    });
  });
};

/**
 * FCM Notification Service - Sends notifications via Firebase Cloud Messaging
 * Supports: PWA (Web Push), React Native, Web
 */
const FCMNotificationService = {
  /**
   * Register device token for a user
   * Called when device registers for push notifications
   */
  async registerDeviceToken(userId, deviceToken, platform = 'web') {
    try {
      // Check if token already exists
      const existing = await queryAsync(
        'SELECT id FROM fcm_device_tokens WHERE user_id = ? AND token = ?',
        [userId, deviceToken]
      );

      if (existing.length > 0) {
        return { success: true, message: 'Token already registered' };
      }

      // Save new token
      await queryAsync(
        'INSERT INTO fcm_device_tokens (user_id, token, platform, created_at) VALUES (?, ?, ?, NOW())',
        [userId, deviceToken, platform]
      );

      console.log(`✅ FCM token registered for user ${userId} (${platform})`);
      return { success: true, message: 'Device token registered', token: deviceToken };
    } catch (error) {
      console.error('❌ Error registering FCM token:', error);
      return { success: false, error: error.message };
    }
  },

  /**
   * Send FCM notification to specific user
   * Sends to all their registered devices
   */
  async sendToUser(userId, notification) {
    try {
      // Get all device tokens for user
      const tokens = await queryAsync(
        'SELECT token, platform FROM fcm_device_tokens WHERE user_id = ?',
        [userId]
      );

      if (tokens.length === 0) {
        console.log(`⚠️ No FCM tokens found for user ${userId}`);
        return { success: false, error: 'No tokens registered for this user' };
      }

      const deviceTokens = tokens.map(t => t.token);
      const result = await this.sendToTokens(deviceTokens, notification);
      
      return result;
    } catch (error) {
      console.error('❌ Error sending FCM notification:', error);
      return { success: false, error: error.message };
    }
  },

  /**
   * Send FCM notification to specific tokens
   * Core method that interacts with Firebase
   */
  async sendToTokens(deviceTokens, notification) {
    try {
      if (!admin.apps.length) {
        return { success: false, error: 'Firebase not configured' };
      }

      // Prepare notification payload
      const message = {
        notification: {
          title: notification.title,
          body: notification.body,
        },
        data: {
          leadId: notification.leadId?.toString() || '',
          type: notification.type || 'notification',
          timestamp: new Date().toISOString(),
        },
        // For web (PWA)
        webpush: {
          notification: {
            title: notification.title,
            body: notification.body,
            icon: '/logo/logo-192.png',
            badge: '/logo/logo-96.png',
            click_action: `${process.env.NEXT_PUBLIC_API_URL || 'http://localhost:3000'}/leads/${notification.leadId}`,
          },
        },
        // For mobile (React Native)
        apns: {
          payload: {
            aps: {
              alert: {
                title: notification.title,
                body: notification.body,
              },
              badge: 1,
            },
          },
        },
      };

      // Send to all tokens
      const responses = await Promise.allSettled(
        deviceTokens.map(token => 
          admin.messaging().send({ ...message, token })
        )
      );

      // Log results
      const successful = responses.filter(r => r.status === 'fulfilled').length;
      const failed = responses.filter(r => r.status === 'rejected').length;

      console.log(`📤 FCM sent to ${deviceTokens.length} tokens (✅ ${successful}, ❌ ${failed})`);

      // Remove failed tokens
      const failedTokens = [];
      responses.forEach((response, index) => {
        if (response.status === 'rejected') {
          const error = response.reason;
          if (error.code === 'messaging/invalid-registration-token' ||
              error.code === 'messaging/registration-token-not-registered') {
            failedTokens.push(deviceTokens[index]);
          }
        }
      });

      if (failedTokens.length > 0) {
        await Promise.all(
          failedTokens.map(token =>
            queryAsync('DELETE FROM fcm_device_tokens WHERE token = ?', [token])
          )
        );
        console.log(`🗑️ Removed ${failedTokens.length} invalid tokens`);
      }

      // Log notification in database
      if (notification.leadId) {
        await queryAsync(
          'INSERT INTO notifications (type, user_id, lead_id, title, body, status) VALUES (?, ?, ?, ?, ?, ?)',
          ['fcm', notification.userId, notification.leadId, notification.title, notification.body, 'sent']
        ).catch(err => console.warn('Could not log notification:', err));
      }

      return {
        success: successful > 0,
        totalSent: deviceTokens.length,
        successful,
        failed,
        removedTokens: failedTokens.length,
      };
    } catch (error) {
      console.error('❌ Error in sendToTokens:', error);
      return { success: false, error: error.message };
    }
  },

  /**
   * Send lead assignment notification
   */
  async sendLeadNotification(userId, leadData) {
    const notification = {
      title: `New Lead: ${leadData.name}`,
      body: leadData.phone || 'New lead assigned to you',
      leadId: leadData.id,
      userId: userId,
      type: 'lead_assignment',
    };

    return this.sendToUser(userId, notification);
  },

  /**
   * Send follow-up reminder notification
   */
  async sendFollowUpNotification(userId, leadData) {
    const notification = {
      title: `Follow-up: ${leadData.name}`,
      body: `Follow-up due: ${leadData.followUpNote || 'Check this lead'}`,
      leadId: leadData.id,
      userId: userId,
      type: 'followup_reminder',
    };

    return this.sendToUser(userId, notification);
  },

  /**
   * Get notification history for user
   */
  async getHistory(userId, limit = 20) {
    try {
      const history = await queryAsync(
        'SELECT * FROM notifications WHERE user_id = ? ORDER BY created_at DESC LIMIT ?',
        [userId, limit]
      );
      return { success: true, data: history };
    } catch (error) {
      console.error('❌ Error fetching notification history:', error);
      return { success: false, error: error.message };
    }
  },

  /**
   * Clean up old tokens (call periodically)
   */
  async cleanupOldTokens(daysOld = 30) {
    try {
      const date = new Date();
      date.setDate(date.getDate() - daysOld);

      const result = await queryAsync(
        'DELETE FROM fcm_device_tokens WHERE updated_at < ? OR (updated_at IS NULL AND created_at < ?)',
        [date, date]
      );

      console.log(`🧹 Cleaned up ${result.affectedRows || 0} old FCM tokens`);
      return { success: true, removed: result.affectedRows };
    } catch (error) {
      console.error('❌ Error cleaning up old tokens:', error);
      return { success: false, error: error.message };
    }
  },
};

module.exports = FCMNotificationService;
